%% Tensor renormalization group and simple update ground-state search
% Author: <https://www.theorie.physik.uni-muenchen.de/lsvondelft/members/sci_mem/seung_sup-lee/index.html 
% Seung-Sup Lee>
%% 
% In this tutorial, we implement the *tensor renormalization group (TRG)* method 
% to compute the correlation functions of a projected entangled-pair state (PEPS) 
% on a square or honeycomb lattice with periodic boundary condition. Also, we 
% study the *simple update* method of finding the ground state of a Heisenberg 
% model on a honeycomb lattice. We will largely refer to Secs. II A and II C of 
% Gu2008 [<https://journals.aps.org/prb/abstract/10.1103/PhysRevB.78.205116 Z.-C. 
% Gu, M. Levin, and X.-G. Wen, Phys. Rev. B *78*, 205116 (2008)> or <https://arxiv.org/abs/0806.3509 
% its arXiv version>] for the TRG, and to Jiang2008 [<https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.101.090603 
% H. C. Jiang, Z. Y. Weng, and T. Xiang, Phys. Rev. Lett. *101*, 090603 (2008)> 
% or <https://arxiv.org/abs/0806.3719 its arXiv version>].
%% Exercise (a): Complete |TRG_Square_Ex.m| and |TRG_Honeycomb_Ex.m|
% Complete two functions zipped together with this document. They are desinged 
% to perform the TRG on a square lattice and on a honeycomb lattice, respectively. 
% *Complete the parts which are enclosed by the comments |TODO - Exercise (a)|.* 
% Once you complete the function, you can follow the first demonstration below.
% 
% There are several remarks.
%% 
% * *Leg order convention*: The function |TRG_Square_Ex.m| receives the input 
% ket tensors |A| and |B| whose legs are organized as in the previous tutorial 
% T12: left-up-(physical)-down-right. (For double tensors, or equivalently transfer 
% operators, the physical leg is absent.) However, inside the function, the legs 
% are permuted to be in counter-clockwise order, so that the legs with the same 
% indices are contracted together. This leg order convention largely simplifies 
% code writing. On the other hand, the function |TRG_Honeycomb_Ex.m| receives 
% the input of local ket tensors whose legs are already organized in the counter-clockwise 
% order. 
% * *Norm of tensors*: At each RG step, the number of sites in the lattice decreases 
% by 2 in the square lattice case and by 3 in the honeycomb lattice case. Rephrasing, 
% the number of physical sites represented by a single coarse-grained tensor increases 
% by 2 and by 3, respectively, after each RG step. Without proper rescaling, the 
% norm of the tensor may increase as a exponential of an exponential (!) of the 
% RG step iteration. So the total norm often numerically diverges after a few 
% steps. To deal with this numerically, the functions compute the squared norm 
% per lattice site, i.e., $( \langle\Psi | \Psi \rangle )^{1/N}$ where $| \Psi 
% \rangle$ is the total state and $N$ is the number of lattice sites.
% * *Last contraction step*: The RG steps are taken until we have a PEPS which 
% is small enough to contract exactly. For the square lattice case, it is 2 by 
% 2; for the honeycomb lattice case, it is of 2 sites.
% * There is a new function |PEPS/contPlaq.m| which contracts tensors arranged 
% around a plaquette. At each RG step of the TRG, the tensors are decomposed via 
% the SVD. Then the decomposed tensors, placed around a plaquette, are contracted. 
% Thus using this function would simplify code writing.
%% Correlation function of the RVB state
% Let's compute the spin-spin correlation $\langle \hat{S}_{(i,j)}^z \hat{S}_{(i,j+1)}^z 
% \rangle$ for the nearest neighbors $(i, j)$ and $(i, j+1)$ for the RVB state 
% on a square lattice. (The honeycomb lattice code, |TRG_Honeycomb_Ex.m|, will 
% be used in the next demonstration below.) The RVB state is considered in the 
% last tutorial T12. While in T12 the small system (a strip of a few hundred sites) 
% with open boundary condition is considered, here we consider very large systems 
% with periodic boundary condition.
% 
% To obtain the PEPS representation of the RVB state, we adopt the lines used 
% in the demonstration of T12.

clear
% left(in)-right(out) for horizontal valence bond,
% or up(in)-down(out) for vertical valence bond
VB = blkdiag(1,[0,1;-1,0]);

P = zeros(3,3,2,3,3); % left(in)-up(in)-physical(in)-down(out)-right(out)

for it1 = (1:3) % left (in)
    for it2 = (1:3) % up (out)
        for it3 = (1:3) % down (out)
            for it4 = (1:3) % right (out)
                it_tot = [it1 it2 it3 it4];
                % spin-up in the physical space
                if (sum(it_tot == 1) == 3) && (sum(it_tot == 2) == 1)
                    P(it1,it2,1,it3,it4) = 1;
                end
                
                % spin-down in the physical space
                if (sum(it_tot == 1) == 3) && (sum(it_tot == 3) == 1)
                    P(it1,it2,2,it3,it4) = 1;
                end
            end
        end
    end
end

A = contract(P,5,4,VB,2,1,[(1:3) 5 4]);
A = contract(A,5,5,VB,2,1);

[S,I] = getLocalSpace('Spin',1/2);
Sz = squeeze(S(:,3,:));
%% 
% Compute the spin-spin correlation function.

Nkeep = 50;
log2L = 10; % lattice size = 2^log2L by 2^log2L

[Nsq,SzSzval] = TRG_Square_Ex (A,A,log2L,Nkeep,Sz,Sz);
%% Exercise (b): Complete |SimpleUp_Honeycomb_Ex.m|
% There is a function |SimpleUp_Honeycomb_Ex.m| zipped together with this document. 
% The function is designed to find the ground state, as a PEPS, of an infinite 
% system on a honeycomb lattice whose Hamiltonian consists of nearest-neighbor 
% interaction terms. Keep in mind that the simple update method is the generalization 
% of the infinite time-evolving block decimation (iTEBD) method, a 1D method covered 
% in the previous tutorial T08, to two dimensions.
% 
% *Complete the parts which are enclosed by the comments |TODO - Exercise (b)|.* 
% Once you complete the function, you can follow the second demonstration below.
%% The ground state of the Heisenberg model on a honeycomb lattice
% Let's compute the ground state of the Heisenberg model on a honeycomb lattice, 
% by using the simple update method. 

clear

% % system parameters
z = 3; % coordination number = number of nearest neighbors
[S,I] = getLocalSpace('Spin',1/2); % local operators
% Heisenberg interaction as two-site gate S*S
HSS = contract(S,3,2,permute(conj(S),[3 2 1]),3,2);

% % simple update parameters
Nkeep = 8;
beta_init = 1e-1; % initial imaginary time step size
beta_fin = 1e-4; % final imaginary time step size
Nstep = 1e4; % number of imaginary time steps
betas = beta_init*((beta_fin/beta_init).^linspace(0,1,Nstep));
% discrete imaginary time steps; decays slowly but exponentially
%% 
% Here we use very small maximum bond dimension, |Nkeep = 8|, contrary to previously 
% demonstrated methods. It is because the computational time complexity of the 
% simple update method, $O(D^{z+1}d^2 + D^3 d^6)$, increases quickly as the bond 
% dimension $D$ increases. Here $z$ is coordination number and $d$ is local space 
% dimension.
% 
% The simple update method is the generalization of the iTEBD for two dimensions. 
% It considers an ansatz wavefunction consists of $\Gamma$ tensors (having physical 
% legs) and $\Lambda$ tensors (connecting $\Gamma$ tensors, carrying the information 
% of entanglement at the bonds). We initialize these tensors with random numbers.

% initialize Lambda and Gamma tensors
Lambda = cell(1,z); % for the bonds in three different directions
Lambda(:) = {rand(Nkeep,1)};
GA = rand([Nkeep*[1 1 1],size(I,2)]); % Gamma for sublattice A sites
GB = GA; % Gamma for sublattice B sites
%% 
% We use the leg order convention for the $\Gamma$ tensors such that the physical 
% leg is placed at the last; see the documentation of |SimpleUp_Honeycomb_Ex.m| 
% and |TRG_Honeycomb_Ex.m|.
% 
% Run the simple update method.

[Lambda,GA,GB,Es] = SimpleUp_Honeycomb_Ex (Lambda,GA,GB,HSS,Nkeep,betas);
%% 
% Note that the 4th output |Es| is the energy across a bond, measured just before 
% applying imaginary time evolution step onto the bond. The "environment" tensors, 
% which are not directly associated with the bond of interest, are *not* in any 
% canonical form. Thus this on-the-fly measurement of the energy can be inaccurate.
%% 
% To have a better estimate of the ground-state energy, we use the TRG. For 
% this, we split the singular value tensors, |Lambda|, into two by taking their 
% square roots, and then absorb the square roots to the |Gamma| tensors. Then 
% the ground state is expressed by two rank-4 local tensors |TA| and |TB|.

TA = GA; TB = GB;
for it3 = (1:z)
    TA = contract(TA,z+1,it3,diag(sqrt(Lambda{it3})),2,2,[(1:it3-1),z+1,(it3:z)]);
    TB = contract(TB,z+1,it3,diag(sqrt(Lambda{it3})),2,1,[(1:it3-1),z+1,(it3:z)]);
end
%% 
% Then run the TRG.

E_GS = zeros(3,1); % for three different bond directions
rgstep = 11; % large enough number, to estimate the value in thermodynamic limit
Nkeep2 = 50; % larger bond dimension than in the simple update
[~,E_GS(1)] = TRG_Honeycomb_Ex (TA,TB, ...
    rgstep,Nkeep2,S,permute(conj(S),[3 2 1]));
%% 
% Note that the last input to |TRG_Honeycomb_Ex| is the Hermitian conjugate 
% of spin operator.

[~,E_GS(2)] = ...
    TRG_Honeycomb_Ex (permute(TA,[2 3 1 4]),permute(TB,[2 3 1 4]), ...
    rgstep,Nkeep2,S,permute(conj(S),[3 2 1]));
[~,E_GS(3)] = ...
    TRG_Honeycomb_Ex (permute(TA,[3 1 2 4]),permute(TB,[3 1 2 4]), ...
    rgstep,Nkeep2,S,permute(conj(S),[3 2 1]));
%% 
% To obtain the energy across different bond directions, we have rotated the 
% tensors |TA| and |TB| accordingly.

disp(E_GS)
%% 
% These energy values are close to each other, but not exactly the same. It 
% is similar as that in the iTEBD calculation the even- and odd-bond energies 
% are different and oscillating with iterations. So we average them.

disp(mean(E_GS))
%% 
% This is the ground-state energy *per bond*. On the other hand, Jiang2008 provides 
% the value *per site*. To make comparison, we multiply 3/2 to the value per bond.

disp(mean(E_GS)*(3/2))
%% 
% The error against the published result -0.5506 (for the bond dimension $D=8$, 
% in Jiang2008) is smaller than 0.4%.