function [Nsq,varargout] = TRG_Honeycomb_Ex (A,B,rgstep,Nkeep,varargin)
% < Description >
%
% Nsq         = TRG_Honeycomb_Ex (A,B,rgstep,Nkeep)
% [Nsq,OOval] = TRG_Honeycomb_Ex (A,B,rgstep,Nkeep,OA,OB)
%
% Compute the squared norm (e.g., < \Psi | \Psi >) per site, of a projected
% entangled-pair state (PEPS) on a square lattice of size L x L with
% periodic boundary condition, by using the tensor renormalization group
% (TRG) method. Here the PEPS is built from a unit cell of two sites,
% represented by local ket tensors A and B, which correspond to two
% bipartition sublattices, respectively. If local operators OA and OB are
% given, this function computes the nearest-neighbour correlator of OA and
% OB, with respect to the PEPS.
%
% Contract the tensors on a honeycomb lattice with open boundary condition,
% by using the tensor-entanglement renormalization group method. Here we
% considere the unit cell of two tensors A and B. If operators OA and OB
% are given, this function computes the nearest-neighbour correlator of OA
% and OB.
%
% < Input >
% A, B : [rank-4 tensors] State tensors for each sublattice. The leg order
%       convention is counter-clockwise: 
%
%     \2            3/
%      \   1    1   /    (Physical legs are the last, i.e., the 4th;
%       A -------- B      they are hidden for brevity)
%      /            \ 
%     /3            2\
%
%       The n-th leg of A contracts with the n-th leg of B for n = 1,2,3.
% rgstep : [integer] # of RG steps. Here the original lattice has
%       2*(3^rgstep) number of sites.
% Nkeep : [numeric] Maximum bond dimension.
% OA, OB : [rank-2 or rank-3 tensors] (Optional) The operators acting onto
%       nearest-neighbor sites. OA (OB) acts on a site whose ket state is
%       represented by A (B). The 1st legs of OA and OB act on the bra, and
%       the last legs on the ket. If they are rank-3, the 2nd legs of A and
%       B are to be contracted each other.
%
% < Output >
% Nsq : [numeric] The squared norm of the PEPS, per lattice site. The
%       total norm of the PEPS is given by Nsq^(2*(3^rgstep)).
% OOval : [numeric] (Optional) Only for when OA and OB are given. The
%       correlation function of local operators OA and OB acting onto
%       nearest-neighbor sites.
%
% Written by S.Lee (Jul.18,2017)
% Updated by S.Lee (Jul.08,2019): Revised for SoSe 2019.

tobj = tic2;
isOp = false; % default: no operator -> not compute correlator

% parsing optional input
if nargin > 4
    isOp = true;
    OA = varargin{1};
    OB = varargin{2};
end

disp(['TERG on a honeycomb lattice: rgstep = ',sprintf('%g',rgstep), ...
    ', Nkeep = ',sprintf('%g',Nkeep)]);

% make double tensor (transfer operator)
z = 3; % coordination number = number of nearest neighbours
ids = [(1:z);z+(1:z)]; % leg permutation
gA = contract(A,z+1,z+1,conj(A),z+1,z+1,ids(:));
gB = contract(B,z+1,z+1,conj(B),z+1,z+1,ids(:));

dimA = ones(2,z); dimB = ones(2,z); % bond dimensions
dimA(1:ndims(gA)) = size(gA);
dimB(1:ndims(gB)) = size(gB);

gA = reshape(gA,prod(dimA,1)); % make fat legs
gB = reshape(gB,prod(dimB,1));

if isOp
    if size(OA,3) > 1 % if OA is rank-3
        % permute the legs so that bra - ket - operator
        OA = permute(OA,[1 3 2]);
    end
    if size(OB,3) > 1 % if OB is rank-3
        % permute the legs so that bra - ket - operator
        OB = permute(OB,[1 3 2]);
    end 
    
    % % % % TODO - Exercise (a)  (Start) % % % %
    % (Hint: refer to the code in TRG_Square_Ex.m)
    
    % permute legs to fuse bond legs
    

    % fuse bond legs by reshaping
    

    % % % % TODO - Exercise (a)  (End) % % % %
    
    Timp = {gOA,gOB}; % impurity tensors
end

Nsq = 1; % initialize with 1

disptime('Start');

for it1 = (1:rgstep)
    Tnorm = sqrt(norm(gA(:))*norm(gB(:)));
    Nsq = Nsq*exp(log(Tnorm)/(3^(it1-1))); % each tensor (gA, gB, ...) represents 3^(it1-1) number of lattice sites
    
    % normalize tensors
    gA = gA/Tnorm;
    gB = gB/Tnorm;
    if isOp
        for it2 = (1:numel(Timp))
            Timp{it2} = Timp{it2}/Tnorm;
        end
    end
    
    % % % % TODO - Exercise (a)  (Start) % % % %
    % contraction of gA and gB, along three different directions
    


    % SVD the contractions of gA and gB. And take the square root of
    % singular values, and contract the square root with the isometries.
    


    % contract the tensors around a plaquette to make the coarse-grained uniform tensor
    

    % after each RG step, the whole lattice rotated clockwise by 90 degree
    
    if isOp
        % when local operators to be measured are given, coarse-grain
        % "impurity" tensors also
        
        
    end
    % % % % TODO - Exercise (a)  (End) % % % %
    
    disptime(['#',sprintf('%02i/%02i',[it1,rgstep])]);
end

% now we have two tensors; contract exactly
res = contract(gA,3,(1:3),gB,3,(1:3));
Nsq = Nsq*exp(log(res)/(2*(3^rgstep)));

if isOp
    % only the ratio of the contraction results of final small tensor
    % networks matters
    varargout{1} = contract(Timp{1},3,(1:3),Timp{2},3,(1:3))/res;
end

strs = ['Squared norm per site = ',sprintf('%.6g',Nsq)];
if isOp
    strs = [strs,', Correlation func. = ',sprintf('%.6g',varargout{1})];
end
disptime(strs);
toc2(tobj,'-v');

end