function [Lambda,GA,GB,Es] = SimpleUp_Honeycomb_Ex (Lambda,GA,GB,H,Nkeep,betas)
% < Description >
%
% [Lambda,GA,GB,Es] = SimpleUp_Honeycomb_Ex (Lambda,GA,GB,H,Nkeep,betas)
%
% Find the ground state, as a projected entangled-pair state (PEPS), of an
% infinite system on a honeycomb lattice whose Hamiltonian consists of
% nearest-neighbor interaction terms H. The state is described by a unit
% cell of two sites, and represented by the Gamma-Lambda notation
% generalized to two dimensions. Similarly to the infinite time-evolving
% block decimation (iTEBD) algorithm of finding the ground-state of
% infinite one-dimensional system, we use the imaginary time evolution to
% find the ground state.
%
% < Input >
% Lambda : [cell array] Each cell Lambda{n} is the vector of singular
%       values that describe the bonds connecting Gamma tensors.
% GA, GB : [rank-4 tensor] Gamma tensors that have physical legs onto A and
%       B sublattice sites, respectively. The detailed description of
%       Lambda{n}, GA, and GB is as below.
%
%  \1                                 2/
%   \                                 /
%   Lambda{2}                      Lambda{3}
%     \2                           1/
%      \2  1                   1   /3       (Physical legs of GA and GB are
%       GA ----- Lambda{1} ----- GB           the last, i.e., the 4th; they
%      /3     2             1      \2         are hidden for brevity)
%     /2                            \1
%   Lambda{3}                      Lambda{2}
%   /                                 \
%  /1                                  \2
%
%       The numbers next to the legs are the leg order. The first legs of
%       Lambda{..} head towards GB, and the second legs towards GA. The
%       legs of GA and GB are ordered in counter-clockwise way, and the
%       n-th legs of GA and GB contract together.
% H : [rank-4 tensor] Two-site interaction term. The leg order convention
%       is:
%      2      4
%      ^      ^
%      |      |
%   [     H      ]
%      |      |
%      ^      ^
%      1      3
% Nkeep : [numeric] Maximum bond dimension.
% betas : [numeric vector] The vector of imaginary time step sizes.
%
% < Output >
% Lambda, GA, GB : The result tensors after imaginary time evolution.
% Es : [matrix] Measured value of two-site term H across each bond *before*
%       applying the imaginary time evolution to the bond. Each column is
%       the value at each iteration, each row is the value for each
%       Lambda{n}.
%
% Written by S.Lee (Jul.08,2019)

tobj = tic2;
ldim = size(H,1); % local space dimension
z = numel(Lambda); % coordination number = number of nearest neighbors

% diagonalize the Hamiltonian to exponentiate
Hmat = reshape(permute(H,[1 3 2 4]),(ldim^2)*[1 1]); % matrix representation
[VH,DH] = eig((Hmat+Hmat')/2);
DH = diag(DH);

% record of energy measurement
Es = zeros(z,numel(betas));

disptime('Start');

for it1 = (1:numel(betas))
    for it2 = (1:z) % index for the bond (or Lambda) to which the imaginary
                    % time evolution is applied
        % contract Lambda's and Gamma
        TA = GA; TB = GB;
        for it3 = (1:z)
            if it3 ~= it2 % Lambda{it2} will be contracted later
                TA = contract(TA,z+1,it3,diag(Lambda{it3}),2,2,[(1:it3-1),z+1,(it3:z)]);
                TB = contract(TB,z+1,it3,diag(Lambda{it3}),2,1,[(1:it3-1),z+1,(it3:z)]);
            end
        end
        
        % % % % TODO - Exercise (b)  (Start) % % % %
        % % Bond projection
        % decompose tensors TA and TB (via SVD) so that their bond legs (to
        % be contracted with Lambda{it2}) and the physical legs are
        % associated with the decomposed parts of TA and TB, while the
        % other bond legs are associated with the other decomposed parts.
        % Note: Should be no truncation here!
        

        % combine singular values and isometries having physical legs
        
        % leg order: 1 (virtual bond to UA), 2 (virtual bond to UB), 3
        % (physical bond of GA), 4 (physical bond of GB)
        
        % normalize T
        
        % measure energy
        

        % imaginary time Trotter step
        

        % decompose the projected bond tensor
        
        % normalize singular value (to normalize the total state)
        
        % rearrange leg order: (leg to be contracted to UA/UB)-(it2-th
        % leg)-(physical bond)
        

        % divide the singluar values (that are associated with the other
        % bonds than Lambda{it2})
        
        
        % % % % TODO - Exercise (b)  (End) % % % %
    end
    
    if (mod(it1,3e3) == 0) || (it1 == numel(betas))
        disptime(['#',sprintf('%i/%i',[it1 numel(betas)]), ...
            ', Measured energy = ',sprintf('%.6g',Es(it2,it1))]);
    end
end

toc2(tobj,'-v');

end