%% NRG: Iterative diagonalization and energy flow
% Author: <https://www.theorie.physik.uni-muenchen.de/lsvondelft/members/sci_mem/seung_sup-lee/index.html 
% Seung-Sup Lee>
% 
% Numerical renormalization group (NRG) is a method for solving quantum impurity 
% systems. <https://en.wikipedia.org/wiki/Kenneth_G._Wilson Kenneth G. Wilson> 
% has itnvented NRG to solve the Kondo problem which was not solvable then. The 
% invention of NRG, indeed, is a part of his Nobel prize citation. NRG is an ancestor 
% of all numerical methods having "renormalization group" in their names.
% 
% We have already covered the iterative diagonalization and the Lanczos tridiagonaliation 
% in earlier tutorials. So one can recycle the solutions of those earlier tutorials 
% to solve the exercises for this tutorial.
%% Logarithmic discretization of bath
% The starting point of the methods we covered before, such as DMRG and iTEBD, 
% is the Hamiltonian of a system which is already discrete, such as chain. On 
% the other hand, an NRG calculation starts from discretizing the continuous system.
% 
% Here we consider an example of quantum impurity system, where the impurity 
% is a spinful fermionic level and the bath consists of non-interacting spinful 
% fermions. The Hamiltonian is given by
% 
% $$H = H_\mathrm{imp} [ \hat{d}_s, \hat{d}_s^\dagger ] + H_\mathrm{hyb} [ \hat{d}_s, 
% \hat{d}_s^\dagger , \hat{c}_{ks}, \hat{c}_{ks}^\dagger  ] + H_\mathrm{bath} 
% [ \hat{c}_{ks}, \hat{c}_{ks}^\dagger ],$$
% 
% $$H_\mathrm{hyb} = \sum_k \sum_{s = {\uparrow},{\downarrow}} v_k (\hat{d}_s^\dagger 
% \hat{c}_{ks} + \hat{c}_{ks}^\dagger \hat{d}_s ),$$
% 
% $$H_\mathrm{bath} = \sum_k \sum_{s = {\uparrow},{\downarrow}} \epsilon_k \hat{c}_{ks}^\dagger 
% \hat{c}_{ks} ,$$
% 
% where $s = {\uparrow},{\downarrow}$ is spin, $\epsilon_k$ is the energy of 
% bath fermion of momentum $k$, and $v_k$ is the coupling amplitude between the 
% impurity level (to which a particle of spin $s$ is added by applying $\hat{d}_s^\dagger$) 
% and the bath level of momentum $k$. The coupling between the impurity and the 
% bath is characterized by the hybridization function,
% 
% $$\Delta (\omega) = \sum_k v_k^2 \, \delta (\omega - \epsilon_k).$$
% 
% In this demonstration, we choose the Anderson impurity and the "box-shaped" 
% hybridization function,
% 
% $$H_\mathrm{imp} = U \hat{n}_{d\uparrow} \hat{n}_{d\downarrow} + \epsilon_d 
% (\hat{n}_{d\uparrow} + \hat{n}_{d\downarrow}) ,$$
% 
% $$\Delta (\omega) = \frac{\Gamma}{\pi} \Theta (D - \omega),$$
% 
% where $\hat{n}_{ds} = \hat{d}_{s}^\dagger \hat{d}_s$ is a number operator 
% at the impurity, $U$ is the local Coulomb interaction, $\epsilon_d$ is the impurity 
% energy level, $\Gamma$ is hybridzation strength, and $D$ is the half-bandwidth 
% of the bath. This case is called *single-impurity Anderson model (SIAM)*. Throughout 
% this tutorial, we set $D=1$ as an energy unit, without loss of generality. 
% 
% The "box-shaped" hybridization function would look like:
% 
% 
% 
% To discretize this, we consider a logarithmic discretization parameter $\Lambda 
% > 1$. The logarithmic frequency grid $\pm \Lambda^{-n}$ splits the whole bandwidth 
% $\omega \in [-1, 1]$ into the intervals $\mathcal{I}_{\pm n}$ that are defined 
% by $\mathcal{I}_{+n} = [\Lambda^{-n}, \Lambda^{-n+1}]$ and $\mathcal{I}_{-n} 
% = [-\Lambda^{-n+1}, -\Lambda^{-n}]$.
% 
% 
% 
% The part of the bath on each interval $\mathcal{I}_{\pm n}$ is replaced by 
% a discrete level at $\omega = \xi_{\pm n}$ that is coupled to the impurity with 
% amplitude $\gamma_{\pm n}$.
% 
% 
% 
% The discretized Hamiltonian is written by
% 
% $$H_\mathrm{bath} \mapsto H_\mathrm{bath}^\mathrm{star} = \sum_{\pm n} \sum_s 
% \xi_{\pm n} \hat{a}_{\pm n, s}^\dagger \hat{a}_{\pm n, s} ,$$
% 
% $$H_\mathrm{hyb} \mapsto H_\mathrm{hyb}^\mathrm{star} = \sum_{\pm n} \sum_s 
% \gamma_{\pm n} ( \hat{d}_s^\dagger \hat{a}_{\pm n, s} + \hat{a}_{\pm n, s}^\dagger  
% \hat{d}_s ) .$$
% 
% Each discretized bath level (to which a particle of spin $s$ is added by applying 
% $\hat{a}_{\pm n, s}^\dagger$) represents the part of the bath on an interval 
% $\mathcal{I}_{\pm n}$. Therefore the coupling strength of the level should be 
% the same as the integrated hybridazation strength over the inteval,
% 
% $$\gamma_{\pm n}^2 = \int_{\mathcal{I}_{\pm n}} \mathrm{d}\omega \, \Delta 
% (\omega) .$$
% 
% While the coupling amplitude $\gamma_{\pm n}$ is unambigously determined, 
% there are several different ways to determine the discretized level position 
% $\xi_{\pm n}$. Here we use the Campo–Oliveira scheme [<https://journals.aps.org/prb/abstract/10.1103/PhysRevB.72.104432 
% V. L. Campo and L. N. Oliveira, Phys. Rev. B *72*, 104432 (2005)>], which defines 
% the level position $\xi_{\pm n}$ as
% 
% $$\xi_{\pm n} = \frac{ \int_{\mathcal{I}_{\pm n}} \mathrm{d}\omega \, \Delta(\omega) 
% }{ \int_{\mathcal{I}_{\pm n}} \mathrm{d}\omega \, \frac{\Delta(\omega)}{\omega} 
% } .$$
% 
% (This way of determining $\xi_{\pm n}$ is better than Wilson's original way. 
% There is a more advanced scheme, but it is harder to implement. That's why we 
% use the Campo–Oliveira scheme here.)
% 
% The discretized Hamiltonian $H_\mathrm{imp} + H_\mathrm{hyb}^\mathrm{star} 
% + H_\mathrm{bath}^\mathrm{star}$ is so-called *star-geometry Hamiltonian.* The 
% impurity level and the discretized bath levels (as vertices of a graph), which 
% are coupled via hopping (as edges of the graph), can be depicted as a star graph.
%% Lanczos tridiagonalization
% The star-geometry Hamiltonian is $H_\mathrm{imp} + H_\mathrm{hyb}^\mathrm{star} 
% + H_\mathrm{bath}^\mathrm{star}$ is mapped onto the *Wilson chain Hamiltonian*, 
% via the Lanczos tridiagonalization. The Lanczos method (which is implemented, 
% for example, in |DMRG/eigs_1site.m| in the context of DMRG) first constructs 
% a tridiagonal matrix representation of the input matrix constrained within the 
% Krylov space, and then diagonalizes the tridiagonal matrix to obtain the ground 
% state. The Lanczos tridiagonalization indicates the first part of this process. 
% Here in the mapping onto the Wilson chain, we consider the tridiagonal matrix 
% representation of the quadratic (i.e., single-particle) terms of the bath and 
% the hybridization, without the quartic (i.e., interacting) impurity Hamiltonian.
% 
% The Wilson chain Hamiltonian for the SIAM is given by
% 
% $$H_\mathrm{SIAM}^\mathrm{chain} = H_\mathrm{imp} + H_\mathrm{bath}^\mathrm{chain} 
% + H_\mathrm{hyb}^\mathrm{chain},$$
% 
% $$H_\mathrm{imp} = U \hat{n}_{d\uparrow} \hat{n}_{d\downarrow} + \epsilon_d 
% (\hat{n}_{d\uparrow} + \hat{n}_{d\downarrow}),$$
% 
% $$H_\mathrm{bath}^\mathrm{chain} = \sum_{\ell \geq 0} \sum_{s = \uparrow,\downarrow} 
% t_\ell \left( \hat{f}_{\ell, s}^\dagger \, \hat{f}_{\ell+1,s} + \hat{f}_{\ell+1,s}^\dagger 
% \, \hat{f}_{\ell, s} \right),$$
% 
% $$H_\mathrm{hyb} ^\mathrm{chain}= \sum_s t_\mathrm{imp} \left( \hat{d}_s^\dagger 
% \hat{f}_{0,s} + \hat{f}_{0,s}^\dagger \hat{d}_s \right).$$
% 
% Note that the impurity Hamiltonian $H_\mathrm{imp}$ is not changed along the 
% logarithmic discretization and the tridiagonalizatoin. The Wilson chain is in 
% principle semi-infinite, but in practice we consider a large but finite length. 
% The length $N$ sets in the minimum energy scale $\sim \Lambda^{-N/2}$ to consider.
% 
% We will solve this one-dimensional system with the iterative diagonalization.
%% Exercise (a): Complete |doCLD_Ex.m| for the logarithmic discretization and the Lanczos tridiagonalization
% Complete the function |doCLD_Ex.m| included in the same |.zip| file with this 
% document. Its subfunction |doCLD_1side| performs the logarithmic discretization. 
% *Complete the part which are enclosed by the comments |TODO - Exercise (a)*|, 
% within the main function for the Lanczos tridiagonalization.
%% 
% Once you solve Exercise 1, you can run this demonstration for verifying your 
% impelementation.

clear

Gamma = 8e-5*pi; % hybridization strength

% NRG parameters
Lambda = 2.5; % discretization parameter
N = 55; % length of the Wilson chain

[ff,gg] = doCLD_Ex([-1 1],[1 1]*Gamma/pi,Lambda,N);
%% 
% Since MATLAB indexes the array from 1, we need to shift the indices: |ff(1)| 
% corresponds to $t_\mathrm{imp}$ and |ff(n)| for $n > 1$ corresponds to $t_{\ell 
% = n-2}$.
% 
% The hopping amplitudes |ff| decay exponentianlly, while the on-site energies 
% |gg| are zeros up do double precision. To see this, we rescale the values with 
% factors $\Lambda^{n/2}$. [*Quick exercise*: Why $\Lambda^{n/2}$, not $\Lambda^n$? 
% We started from the discretization grid of $\pm \Lambda^{-n}$, so it might look 
% weird to have factor 1/2 in the exponent; but of course, there is a good reason.]

figure;
plot([ff gg].*(Lambda.^((1:numel(ff)).'/2)), ...
    'LineWidth',1);
set(gca,'FontSize',13,'LineWidth',1);
grid on;
xlabel('Index n');
legend({'ff(n) \times \Lambda^{n/2}', ...
    'gg(n) \times \Lambda^{n/2}'}, ...
    'Location','eastoutside');
%% 
% The first elements of |ff| deviate from the exponential dependence, as we 
% see the deviation from the horizontal line. They come from the specific details 
% of the hybridization function. For example, the square of ff(1) is equivalent 
% to the integral of the hybridization,
% 
% $$t_\mathrm{imp}^2 = \int_{-D}^{D} \mathrm{d}\omega \Delta (\omega) = 2 \Gamma 
% D/ \pi.$$

ff(1)^2 - 2*Gamma/pi % D = 1 as energy unit
%% Exercise (b): Complete |NRG_IterDiag_Ex.m| for the iterative diagonalization _a la_ NRG
% Complete the function |NRG_IterDiag_Ex.m| included in the same |.zip| file 
% with this document. This NRG style of the iterative diagonalization differs 
% from the earlier tutorial in that (i) the Hamiltonian is rescaled by the energy 
% scale factors $\sim \Lambda^{-n/2}$ and (ii) the energy eigenvalues are shifted 
% so that the lowest energy eigenvalue becomes zero. Other than these, it is the 
% same iterative diagonalization. *Complete the part which are enclosed by the 
% comments |TODO - Exercise (b).*|
%% 
% As a demonstration of the completed iterative diagonalization, we apply it 
% to the SIAM.

% Hamiltonian parameters
U = 4e-3; % Coulomb interaction at the impurity
epsd = -U/2; % impurity on-site energy

% NRG parameter
Nkeep = 300;

% Construct local operators
[F,Z,S,I] = getLocalSpace('FermionS');

% particle number operator
NF = cat(3,contract(conj(F(:,1,:)),3,[1 2],F(:,1,:),3,[1 2]), ...
           contract(conj(F(:,2,:)),3,[1 2],F(:,2,:),3,[1 2]));
       
% Impurity Hamiltonian
H0 = U*(NF(:,:,1)*NF(:,:,2)) + epsd*(NF(:,:,1)+NF(:,:,2));

% ket tensor for the impurity
A0 = getIdentity(1,2,I,2); % 1 for dummy leg

% iterative diagonalization
Inrg = NRG_IterDiag_Ex (H0,A0,Lambda,ff,F,gg,sum(NF,3),Z,Nkeep);
%% 
% Here each line indicates the information of each iteration step: time stamp, 
% number of the kept states, number of the total states, the largest energy of 
% the kept states, the largest energy of the discarded states.
%% Energy flow diagam
% NRG provides a method to analyze the spectrum obtained along the iterative 
% diagonalization. Let's plot the lowest-lying (many-body) energy levels. We plot 
% the results from even iterations and those from odd iterations separately.

% Energy flow diagram
Eshow = 3; % energy window to show (from 0 to Eshow)

% since we start from A0; consider the step for H0 as 0, i.e., even
Eeven = Inrg.EK(1:2:end);
Eeven = cellfun(@(x) x(x <= Eshow), Eeven, 'UniformOutput', 0);
maxEeven = max(cellfun('prodofsize',Eeven));
Eeven = cellfun(@(x) [x;nan(maxEeven-numel(x),1)], Eeven, 'UniformOutput', 0);
Eeven = cell2mat(Eeven).';

Eodd = Inrg.EK(2:2:end);
Eodd = cellfun(@(x) x(x <= Eshow), Eodd, 'UniformOutput', 0);
maxEodd = max(cellfun('prodofsize',Eodd));
Eodd = cellfun(@(x) [x;nan(maxEodd-numel(x),1)], Eodd, 'UniformOutput', 0);
Eodd = cell2mat(Eodd).';
%% 
% (Note that |cellfun('prodofsize', ...)| is undocumented functionality, which 
% gives the same result as |cellfun(@numel, ...)| but much faster. Refer to <http://undocumentedmatlab.com/blog/cellfun-undocumented-performance-boost 
% this webpage>.)

figure;
% upper panel
subplot(2,1,1);
plot((1:2:numel(Inrg.EK))-1,Eeven,'LineWidth',1);
xlabel('Even iterations');
xlim([0 numel(Inrg.EK)-1]);
ylim([0, Eshow]);
set(gca,'LineWidth',1,'FontSize',13);

% lower panel
subplot(2,1,2);
plot((2:2:numel(Inrg.EK))-1,Eodd,'LineWidth',1);
xlabel('Odd iterations');
xlim([0 numel(Inrg.EK)-1]);
ylim([0, Eshow]);
set(gca,'LineWidth',1,'FontSize',13);
%% 
% (Note that |cellfun('prodofsize', ...)| is undocumented functionality, which 
% gives the same result as |cellfun(@numel, ...)| but much faster. Refer to <http://undocumentedmatlab.com/blog/cellfun-undocumented-performance-boost 
% this webpage>.)
% 
% These plots are called *energy flow diagram* or *finite-size energy spectra*. 
% The name "flow" literally comes from that the lines flow from one regime to 
% the other. There are three regions (iterations 1$-$10; 17$-$25; 35$-$55) connected 
% via two crossovers. These regions correspond to different *fixed points: free 
% orbital, local moment, and strong coupling.* The strong-coupling fixed-point 
% regime exhibits prominent plateau of the energy levels.
%% Exercise (c): Reproduce lowest-lying energies in the strong-coupling regime by fixed-point Hamiltonians
% Let's consider iteration 53 (counting from iteration 0 for the impurity only) 
% in the strong-coupling fixed-point regime.. Their lowet-lying energies, including 
% all degenerate levels, are:

fprintf([sprintf('%.4f, ',Inrg.EK{54}(1:5).'),'\n', ...
    sprintf('%.4f, ',Inrg.EK{54}(6:11).'),'...\n']);
%% 
% We see 1-fold, 4-fold, and 6-fold degeneracies. On the other hand, the energy 
% levels at the next iteration 54 have more degeneracies:

fprintf([sprintf('%.4f, ',Inrg.EK{55}(1:4).'),'\n', ...
    sprintf('%.4f, ',Inrg.EK{55}(5:12).'),'\n', ...
    sprintf('%.4f, ',Inrg.EK{55}(13:20).'),'...\n']);
%% 
% There are 4-fold and 16-fold degeneracies, up to numerical noise of $O(10^{-6})$. 
% *Reproduce these (many-body) energy values by considering strong-coupling fixed-point 
% Hamiltonians.*
% 
% (_Hint_: The fixed-point Hamiltonians are single-particle Hamiltonians, effectively!)
%% Exercise (d): Single-impurity Kondo model
% We can derive the Wilson chain Hamiltonian $H_\mathrm{SIKM}^\mathrm{chain}$ 
% for the single-impurity Kondo model (SIKM), from the chain Hamiltonian of the 
% SIAM $H_\mathrm{SIAM}^\mathrm{chain}$ shown above. By applying the Schrieffer-Wolff 
% transformation to the impurity site (on which $\hat{d}_s$ acts) and the first 
% bath site (on which $\hat{f}_{0,s}$ acts), we obtain
% 
% $$H_\mathrm{SIKM}^\mathrm{chain} = H_\mathrm{exc} + H_\mathrm{bath},$$
% 
% $$H_\mathrm{exc} = 2J \hat{\vec{S}}_d \cdot \hat{\vec{S}}_0 ,$$
% 
% $$H_\mathrm{bath} = \sum_{\ell \geq 0} \sum_{s = \uparrow,\downarrow} t_\ell 
% \left( \hat{f}_{\ell, s}^\dagger \, \hat{f}_{\ell+1,s} + \hat{f}_{\ell+1,s}^\dagger 
% \, \hat{f}_{\ell, s} \right) .$$
% 
% Here $\hat{\vec{S}}_d$ is the spin operator acting on the impurity site,
% 
% $$\hat{\vec{S}}_0 = \sum_{s,s'} \hat{f}_{0,s}^\dagger \frac{\vec{\sigma}_{s,s'}}{2} 
% \hat{f}_{0,s'}$$
% 
% is the spin operator acting on the first bath site, $\vec{\sigma} = [\sigma_x, 
% \sigma_y, \sigma_z]$ is the vector of Pauli matrices, and
% 
% $$J = t_0^2 \left( \frac{1}{-\epsilon_d} + \frac{1}{U + \epsilon_d} \right)$$
% 
% is the Kondo coupling strength. For particle-hole symmetric case $\epsilon_d 
% = -U/2$ which we considered above, it becomes
% 
% $$J = \frac{4 t_0^2}{U} = \frac{8 \Gamma D}{\pi U}.$$
% 
% The bath term $H_\mathrm{bath}$ is the same as in the SIAM case. Note that 
% the impurity site in the SIKM has dimension 2, while that in the SIAM has 4; 
% the doubly occupied and the empty states are "integrated out" by the Schrieffer-Wolff 
% transformation. Refer to <https://journals.aps.org/pr/abstract/10.1103/PhysRev.149.491 
% J. R. Schrieffer and P. A. Wolff, Phys. Rev. *149*, 491 (1966)> for the details 
% of the Schrieffer-Wolff transformation.
% 
% *Perform the iterative diagonalization of this chain Hamiltonian for the SIKM*, 
% with the value of $J$ corresponding to the choice of parameters $U$, $\epsilon_d$, 
% and $\Gamma$ above. (Again $D = 1$ is the energy unit.) Compare the energy flow 
% diagram with the SIAM result.
% 
% (_Hint_: You can do it by changing |H0|, |A0|, and |ff| only, with using the 
% same function |NRG_IterDiag_Ex|!)