function [out] = initMPO(ham_, beta_, nsite_)
% generate an MPO to approximate the density matrix with the 1st-order Trotter decomposition
% < input >
% ham_: a 2-site local Hamiltonian
% beta_: the inverse temperature
% nsite_: # of sites
%
% < output >
% out: a matrix product operator; index ordering: left-down-up-right

  % compute exp^{-beta*ham}
  [sH1,sH2,sH3,sH4] = size(ham_);
  gate = expm(-beta_*reshape(permute(ham_,[1,3,2,4]),[sH1*sH3,sH2*sH4]));
  gate = reshape(gate,[sH1,sH3,sH2,sH4]);

  % 1st-order Trotter decomposition for the nsite-spin chain
  [uH,sH,vH] = svdTr(gate, 4, [1,3], Inf, 1E-16);
  sH = diag(sqrt(sH));
  uH = contract(uH, 3, [3], sH, 2, [1]);
  vH = contract(sH, 2, [2], vH, 3, [1]);
  mpo0 = contract(uH, 3, [2], vH, 3, [2], [3,1,4,2]);
  out = cell(nsite_,1);
  out(:) = {mpo0};
  out{1} = permute(uH, [4,1,2,3]);
  out{end} = vH; 
end
