function [Lambda,Gamma] = canon_iMPS_Ex (Lambda,Gamma)
% < Description >
%
% [Lambda,Gamma] = canon_iMPS_Ex (Lambda,Gamma)
%
% Orthogonalize Vidal's Gamma-Lambda representation of infinite MPS,
% following the method given in [R. Orus & G. Vidal, Phys. Rev. B 78,
% 155117 (2008)]. Here the goal of the orthogonalization is to make the ket
% tensors of Lambda*Gamma type (Gamma*Lambda type) be left-normalized
% (right-normalized), i.e., to bring them into canonical forms.
%
% < Input >
% Lambda : [cell] Each cell contains the column vector of the singular
%       values at each bond. Number of cells, numel(Lambda), means the size
%       of the unit cell.
% Gamma : [cell] Each cell contains a rank-3 tensor associated with each
%       site within an unit cell. Number of cells, numel(Gamma), means the
%       size of the unit cell, and needs to be the same as numel(Lambda).
%       The ket tensor for the unit cell is represented by:
%
% ->-diag(Lambda{end})->-*->-Gamma{1}->-* ... *->-diag(Lambda{end-1})->-*->-Gamma{end}->-*->-diag(Lambda{end})->-  
%  1                   2   1    ^     3         1                     2   1    ^       3   1                   2 
%                               |2                                             |2
%
%       The Lambda's and Gamma's can be given as random tensors. After the
%       imaginary time evoltion, the ket tensor becomes left-normalized (up
%       to some numerical error).
%
% < Output >
% Lambda, Gamma : [cell] Cell arrays of Lambda and Gamma tensors,
%       repectively, after the orthogonalization.
%
% Written by S.Lee (Jun.16,2017); updated by S.Lee (Jun.19,2017)
% Updated by S.Lee (Jun.01,2019): Revised for Sose 2019.
% Updated by S.Lee (Jun.09,2020): Typo fixed.

Lambda = Lambda(:);
Gamma = Gamma(:);
n = numel(Lambda); % number of sites in the unit cell

% % % check the integrity of input
for it = (1:n)
    if ~isvector(Lambda{it})
        error(['ERR: Lambda{',sprintf('%i',it),'} should be vector.']);
    elseif numel(Lambda{mod(it,n)+1}) ~= size(Gamma{it},1)
        error(['ERR: Dimensions for Lambda{',sprintf('%i',mod(it,n)+1),'} and Gamma{', ...
            sprintf('%i',it),'} do not match.']);
    elseif numel(Lambda{it}) ~= size(Gamma{it},3)
        error(['ERR: Dimensions for Lambda{',sprintf('%i',it), ...
            '} and Gamma{',sprintf('%i',it),'} do not match.']);
    end
end
% % % 

% % "Coarse grain" the tensors: contract the tensors for the unit cell
% % altogether. Then the coarse-grained tensor will be orthogonalized.
% Gamma{1}*Lambda{1}* ... * Gamma{end}
T = Gamma{1};
for it = (2:n)
    T = contract(T,it+1,it+1,diag(Lambda{it-1}),2,1);
    T = contract(T,it+1,it+1,Gamma{it},3,1);
end

% ket tensor to compute transfer operator from right
TR = contract(T,n+2,n+2,diag(Lambda{n}),2,1);
% find the dominant eigenvector for the transfer operator from right
XR = canonIMPS_domVec(TR,n+2,0);

% ket tensor to compute transfer operator from left
TL = contract(diag(Lambda{n}),2,2,T,n+2,1);
% find the dominant eigenvector for the transfer operator from left
XL = canonIMPS_domVec(TL,n+2,1);

% % % % TODO - Exercise 4 (Start) % % % %
% do SVD in Fig. 2(ii) of [R. Orus & G. Vidal, Phys. Rev. B 78, 155117 (2008)]

% orthogonalize the coarse-grained tensor

% % % % TODO - Exercise 4 (End) % % % %

% result
Lambda = cell(n,1);
Gamma = cell(n,1);
Lambda{n} = SX/norm(SX);

if n == 1
    % there is only one site in the unit cell; the job is done
    Gamma{1} = T;
else
    % decompose the orthogonalized coarse-grained tensor into the tensors
    % for individual sites
    
    % contract singular value tensors to the left and right ends, before
    % doing SVD.
    T = contract(diag(Lambda{n}),2,2,T,n+2,1);
    T = contract(T,n+2,n+2,diag(Lambda{n}),2,1);

    for it = (1:n-1)
        if it > 1
            % contract singular value tensor to the left, since the
            % singular value tensor contracted to the left before is
            % factored out by SVD
            T = contract(diag(Lambda{it-1}),2,2,V2,n+2-(it-1),1);
        end
        [U2,S2,V2] = svdTr(T,n+2-(it-1),[1 2],[],[]);
        Lambda{it} = S2/norm(S2);
        Gamma{it} = contract(diag(1./Lambda{mod(it-2,n)+1}),2,2,U2,3,1);
    end
    
    Gamma{end} = contract(V2,3,3,diag(1./Lambda{n}),2,1);
end
    

end

function X = canonIMPS_domVec (T,rankT,isconj)
% Find the tensor X where X*X' originates from the dominant right
% eigenvector of the transfer operator which is
%
%  1                 n+2
%  -->-[     T     ]->--
%      2|   ...   |n+1
%       ^         ^
%       *   ...   *
%       ^         ^
%      2|   ...   |n+1
%  --<-[     T'    ]-<--
%  1                 n+2
%
% isconj == 1 if transfer operator from left, == 0 from right

D = size(T,1);
idT = (2:rankT-1);
T = contract(T,rankT,idT,conj(T),rankT,idT,[1 3 2 4]);
T = reshape(T,[D*D D*D]);
if isconj, T = T.'; end
[V,~] = eigs(T,1,'lm');
V = reshape(V,[D D]);
V = (V+V')/2; % Hermitianize for numerical stability
[V1,D1] = eig(V);
D1 = diag(D1);

% V is equivalent up to overall sign; if the eigenvalues are mostly
% negative, take the overall sign flip
if sum(D1) < 0
    D1 = -D1; 
end

% remove negative eigenvalues, which come from numerical noise
oks = (D1 <= 0);
D1(oks) = [];
V1(:,oks) = [];

X = V1*diag(sqrt(D1));
if isconj, X = X.'; end

end
