%% Expectation values in MPS: Example of AKLT state
% Author: <https://www.theorie.physik.uni-muenchen.de/lsvondelft/members/sci_mem/seung_sup-lee/index.html 
% Seung-Sup Lee>
% 
% *Before following this tutorial, it is necessary to solve Exercise (a) of 
% tutorial "Canonical forms of MPS", since we will use the function |canonForm_Ex.m| 
% to bring the MPSs into canonical forms.*
%% Quick introduction to the Affleck-Kennedy-Lieb-Tasaki (AKLT) state
% The AKLT state is one of the most famous MPSs. Ian Affleck, Tom Kennedy, Elliott 
% Lieb, and Hal Tasaki, who devised this state, showed it to be the exact ground 
% state of the spin-1 chain model in which spins interact via the Heisenberg interaction 
% and biquadratic interaction terms [<https://journals.aps.org/prl/abstract/10.1103/PhysRevLett.59.799 
% I. Affleck _et al._, Phys. Rev. Lett. *59*, 799 (1877)>]. Despite its simple 
% structure (see below), the AKLT state has rich, interesting physical properties. 
% One such property is *symmetry-protected topological order*: there is a finite 
% energy gap between the AKLT state (ground state) and excited states, and the 
% symmetry of the system is not broken. The AKLT state can be generalized to the 
% ground states of the Haldane phase (named after <https://en.wikipedia.org/wiki/Duncan_Haldane 
% Duncan Haldane>) of the spin-1 chain model. Indeed, Haldane's contribution to 
% the theory of topological phases of matter was the main reason to award him 
% the 2016 Nobel Prize in Physics! In this tutorial, we will use the AKLT state 
% as just an example to practice computing expectation values, without explaining 
% further physical implications. The details of the AKLT state will be covered 
% later in the lecture course. Interested students may refer to the <https://en.wikipedia.org/wiki/AKLT_model 
% Wikipedia page>.
% 
% The spin-1 AKLT state can be understood as a product state of valence bonds. 
% First, one associates each spin-1 particle on each physical site with two spin-1/2 
% (virtual) particles, and each of which forms a spin singlet, i.e, a valence 
% bond, with the virtual particles on its neighboring site. The tensor diagram 
% representation of such spin singlet is:
% 
% 
% 
% Here all legs have dimension 2. The physical legs (bottom legs) have dimension 
% 2 since they are associated with spin-1/2's. And the horizontal leg connecting 
% vertices is of size 2 since the Schmidt rank (i.e. number of nonzero Schmidt 
% coefficients, or equivalently, of singular values) is 2. Then we arrange the 
% singlets, and associate the right physical leg of one singlet and the left physical 
% leg of the singlet on its right into each physical site.
% 
% 
% 
% Each site has two spin-1/2's, so the total spin of the site can be either 
% $S = 0$ or $S = 1$. Then we project the local spaces into the $S = 1$subspace.
% 
% 
% 
% The triangles denote the isometries that map the Hilbert space of one spin-1 
% into a subspace of the product space of two spin-1/2's. By contracting the tensors 
% associated with the same site (marked by the dashed-line box), we obtain the 
% rank-3 tensors that constitute the AKLT state as an MPS.
% 
% 
% 
% By construction, all the tensors here are identical. The left and right legs 
% of each tensor have dimension 2, which corresponds to the Schmidt rank 2 of 
% the spin singlet. The physical (bottom) legs have dimension 3, for the spin-1's, 
% with index values 1, 2, and 3 representing the local spins states $| S_z = +1 
% \rangle, | S_z = 0 \rangle,$ and $| S_z = -1 \rangle$,  respectively.
%% Generate the AKLT state
% We will generate the AKLT state on a finite chain of length |N|. The tensor 
% at each bulk site (i.e., any site except the left- and rightmost sites) is defined 
% as 3-dimensional array, |AKLT|:

clear

AKLT = zeros(2,3,2);
% local spin Sz = -1
AKLT(2,1,1) = -sqrt(2/3);
% local spin Sz = 0
AKLT(1,2,1) = -1/sqrt(3);
AKLT(2,2,2) = +1/sqrt(3);
% local spin Sz = +1
AKLT(1,3,2) = sqrt(2/3);

N = 50; % number of sites
M = cell(1,N); % MPS
M(:) = {AKLT};
%% 
% As we are considering a finite system with open boundary condition, the left- 
% and rightmost legs of the MPS should have dimension 1, to represent a single 
% global quantum state. Therefore, we project the space of the left leg of |M{1}| 
% and the space of the right leg of |M{end}| onto the subspaces of size 1 for 
% each leg. Since the left and right legs have dimension 2, there are $4 = 2 \times 
% 2$ different global states, which are linearly independent. We'll denote them 
% by $| \psi (\alpha, \beta) \rangle$ with $\alpha = 1,2$ is chosen for the left 
% leg of |M{1}| and $\beta = 1,2$ for the right leg of |M{end}|. For example, 
% we obtain $|\psi (1,1) \rangle$ by:

M{1} = M{1}(1,:,:);
M{end} = M{end}(:,:,1);
%% 
% [*Warm-up exercise*: Draw the tensor network diagram of the valence bonds 
% and the isometries, for a finite MPS with open boundary condition. Observe that 
% the projection of the leftmost and rightmost legs corresponds to a *boundary 
% condition* which fixes the states of the virtual spin-1/2 particles at the left 
% and right ends.]
% 
% Let us check that the bulk tensors are *both left- and right-normalized at 
% the same time*.
% 
% 

% check whether left-normalized
T = contract(conj(AKLT),3,[1 2],AKLT,3,[1 2]);
disp(T - eye(size(T))); % all zeros
% check whether right-normalized
T = contract(conj(AKLT),3,[2 3],AKLT,3,[2 3]);
disp(T - eye(size(T))); % all zeros
%% 
% Note that the bra tensor is obtained as the complex conjugate of the ket tensor, 
% via |conj|. Of course, |M{1}| is only right-normalized and |M{end}| is only 
% left-normalized, since their left/right legs are projected by boundary condition.

T = contract(conj(M{1}),3,[1 2],M{1},3,[1 2]);
disp(T); % not left-normalized
T = contract(conj(M{1}),3,[2 3],M{1},3,[2 3]);
disp(T); % right-normalized
T = contract(conj(M{end}),3,[1 2],M{end},3,[1 2]);
disp(T); % left-normalized
T = contract(conj(M{end}),3,[2 3],M{end},3,[2 3]);
disp(T); % not right-normalized
%% 
% For computing the expectation values, it is convenient to normalize the MPS. 
% (Otherwise, one needs to divide the expectation value with the square of the 
% norm of the MPS.) Let's bring the MPS into its left-canonical form, without 
% loss of generality.

% transform into left-canonical form
[M,S] = canonForm_Ex(M,numel(M));
fprintf('Norm of MPS = %.4g\n',S);
%% 
% Note that the norm is not unity.
%% Magnetization
% We can compute the local magnetization as the MPS expectation value of the 
% spin-z operator $S_{[j],z}$ acting onto site j. Since we associated the local 
% spin states $| S_z = +1 \rangle, | S_z = 0 \rangle,$ and $| S_z = -1 \rangle$ 
% with the indices 1, 2, and 3 within the physical leg space, the spin-z operator 
% can be implemented as:

Sz = diag([+1;0;-1]);
%% 
% To compute the expectation value, we contract the tensors from the left:
% 
% 

j = 10; % index of site on which Sz operator acts

% T: the contraction of bra/ket/Sz tensors
T = 1; % identity in the dummy leg space for the leftmost leg
% leg order of T and Sz:
%   bottom (towards bras) - top (towards kets)
for it = (1:numel(M))
    T = contract(T,2,1,conj(M{it}),3,1);
    if it == j % contract Sz
        T = contract(T,3,2,Sz,2,1,[1 3 2]);
        % permute to have leg order left-physical-right
    end
    T = contract(T,3,[1 2],M{it},3,[1 2]);
end

fprintf('Magnetization at site %i = %.4g\n',j,T);
%% Exercise (a): Magnetization
% In the demonstration above, the code computes the magnetization at only one 
% site, by contracting all the tensors from the left end to the right end. However, 
% as the MPS was already brought into the left-canonical form, one can start from 
% the site $j$, not from the left end. Keeping this in mind, *compute the magnetization 
% for all chain sites*. And compare the result with the exact analytic result, 
% for all different boundary conditions:
% 
% $$\langle \psi(1,1)|S_{[j],z} | \psi(1,1) \rangle = -\langle \psi(2,2)|S_{[j],z} 
% | \psi(2,2) \rangle= \frac{ \left(-\frac{1}{3}\right)^j - \left(-\frac{1}{3}\right)^{N-j+1} 
% }{ \frac{1}{2}\left[ 1+ \left(-\frac{1}{3}\right)^N \right] } $$
% 
% $$\langle \psi(1,2)|S_{[j],z} | \psi(1,2) \rangle = -\langle \psi(2,1)|S_{[j],z} 
% | \psi(2,1) \rangle= \frac{ \left(-\frac{1}{3}\right)^j + \left(-\frac{1}{3}\right)^{N-j+1} 
% }{ \frac{1}{2}\left[ 1-\left(-\frac{1}{3}\right)^N \right] }$$ 
% 
% *Compute the magnetization for all chain sites again, by using the different 
% definition of bulk tensors*:

AKLT = zeros(2,3,2);
% local spin Sz = -1
AKLT(2,1,1) = -sqrt(2/3); % same
% local spin Sz = 0
AKLT(1,2,1) = -1i/sqrt(3); % changed
AKLT(2,2,2) = +1i/sqrt(3); % changed
% local spin Sz = +1
AKLT(1,3,2) = sqrt(2/3); % same
%% 
% And check the results from different bulk tensors are the same. [*Quick question*: 
% Why do they need to be the same?]
%% Exercise (b): Spin-spin correlation
% Compute the correlation function between the spin-z operators at nearest-neighbor 
% sites $j$ and $j+1$. Compare this with the exact analytic result:
% 
% $$\langle \psi(1,1)|S_{[j],z} S_{[j+1],z} | \psi(1,1) \rangle = \langle \psi(2,2)|S_{[j],z} 
% S_{[j+1],z}| \psi(2,2) \rangle= \frac{ \left(-\frac{2}{9}\right) - 2\left(-\frac{1}{3}\right)^N 
% }{ \frac{1}{2}\left[ 1+ \left(-\frac{1}{3}\right)^N \right] }$$
% 
% $$\langle \psi(1,2)|S_{[j],z} S_{[j+1],z} | \psi(1,2) \rangle = \langle \psi(2,1)|S_{[j],z} 
% S_{[j+1],z} | \psi(2,1) \rangle= \frac{ \left(-\frac{2}{9}\right) + 2\left(-\frac{1}{3}\right)^N 
% }{ \frac{1}{2}\left[ 1- \left(-\frac{1}{3}\right)^N \right] }$$